/**
* \file: AilAudioSourceImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Actual implementation of AilAudioSource class (see pImpl idiom)
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*          P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <chrono>
#include <AudioFactory.h>
#include <AudioBackend.h>
#include <AudioTypes.h>
#include <bdcl/BaiduCoreCallbackDealer.h>
#include <bdcl/CCarLifeLib.h>
#include "AilAudioSourceImpl.h"

LOG_IMPORT_CONTEXT(bdcl_audio)

namespace adit { namespace bdcl {

// todo you don't have to pull whole namespaces, especially for std. simplify!
using namespace adit::utility::audio;
using namespace adit::utility;
using namespace std;

#define NUMOFBITSPERBYTE 8

AilAudioSourceImpl::AilAudioSourceImpl(IAditAudioSourceCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
: mCallbacks(inCallbacks), mCallbackDealer(inCallbackDealer), mShuttingDown(false), captureOffset(0), mStarted(false)
{
    // todo why not just give Alsa as an input parameter? or it can be a const string?
    std::string backendLibName("Alsa");
    mAilBackend = Factory::Instance()->createBackend(backendLibName, *this);
}

AilAudioSourceImpl::~AilAudioSourceImpl()
{
    if (mShuttingDown)
    {
        teardown();
    }
}

// todo const string?
void AilAudioSourceImpl::setConfigItem(std::string inKey, std::string inValue)
{
    LOGD_DEBUG((bdcl_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(), inValue.c_str()));
    mConfig.set(inKey, inValue);
}

bool AilAudioSourceImpl::initialize()
{
    if (!mConfig.ResultConfig()) {
        LOG_ERROR((bdcl_audio, "Could not initialize audio source endpoint due to configuration error"));
        return false;
    } else {
        mCallbackDealer->registerVrCaptureAudioCallbacks(mCallbacks);
        return true;
    }
}

void AilAudioSourceImpl::teardown()
{
    mShuttingDown = true;

    if (mStarted)
    {
        captureStop();
    }
    LOGD_DEBUG((bdcl_audio, "teardown()  audio source is down"));
}

bool AilAudioSourceImpl::captureStart()
{
    uint32_t period_samples = mConfig.mPeriodMs * (mConfig.mSampleRate / 1000);

    std::string invalid_playback_device("");
    bool ret_val = true;

    if (!mStarted)
    {
        if(!mConfig.mDisablePrio)
        {
            mAilBackend->setThreadSched(SCHED_FIFO, mConfig.mThreadPrio);
        }

        if(mConfig.mInitToutMs > 0)
        {
            mAilBackend->setInitialTimeout(mConfig.mInitToutMs);
        }

        // below
        LOGD_DEBUG((bdcl_audio, "Device name to be opened for audio source: %s", mConfig.mDeviceName.c_str()));
        /* todo AudioFormat::S16_LE means 16 bit little endian, which is by luck the same as BDCL although it is again
         * copied from AAUTO. Make it configurable or at least define as macro */
        AudioError err = mAilBackend->openStream(mConfig.mDeviceName, invalid_playback_device, AudioFormat::S16_LE,
                mConfig.mSampleRate, mConfig.mNumofChannels, period_samples);
        if (err != AudioError::OK)
        {
            LOG_ERROR((bdcl_audio, "openStream() failed for audio source: error: %d", static_cast<uint32_t>(err)));
            ret_val = false;

        } else {
            LOGD_DEBUG((bdcl_audio, "openStream() successful for audio source"));

            // below
            err = mAilBackend->startStream();
            if (err != AudioError::OK)
            {
                LOG_ERROR((bdcl_audio, "startStream() failed for audio source, error: %d", static_cast<uint32_t>(err)));
                ret_val = false;

            } else {
                LOGD_DEBUG((bdcl_audio, "startStream() successful for audio source"));

                mStarted = true;
            }
        }
    }
    else
    {
        LOGD_DEBUG((bdcl_audio, "startStream()  audio source already started"));
        // todo Send true or false in case audio source was already started?
        ret_val = true;
    }
    return ret_val;
}

bool AilAudioSourceImpl::captureStop()
{
    AudioError err;
    bool ret_val = true;

    if (mStarted) {
        if (mShuttingDown) {
            err = mAilBackend->abortStream();
            if (err != AudioError::OK) {
                LOG_ERROR((bdcl_audio, "Failed to abort / stop audio capture stream: error: %d", static_cast<uint32_t>(err)));
                ret_val = false;
            }
        } else {
            err = mAilBackend->stopStream();
            if (err != AudioError::OK) {
                LOG_ERROR((bdcl_audio, "Failed to abort / stop audio capture stream: error: %d", static_cast<uint32_t>(err)));
                ret_val = false;
            }
        }

        err = mAilBackend->closeStream();
        if (err != AudioError::OK) {
            LOG_ERROR((bdcl_audio, "Failed to close audio capture stream: error: %d", static_cast<uint32_t>(err)));
            ret_val = false;
        }

        mStarted = false;
    } else {
        LOGD_DEBUG((bdcl_audio, "captureStop()  audio source was not started (mStarted: %d)", mStarted));
    }

    return ret_val;
}

void AilAudioSourceImpl::error(const std::string& inData) const
{
    LOG_ERROR((bdcl_audio, "%s", inData.c_str()));
}

void AilAudioSourceImpl::warning(const std::string& inData) const
{
    LOG_WARN((bdcl_audio, "%s", inData.c_str()));
}

void AilAudioSourceImpl::info(const std::string& inData) const
{
    LOG_INFO((bdcl_audio, "%s", inData.c_str()));
}

void AilAudioSourceImpl::debug(const std::string& inData) const
{
    LOGD_DEBUG((bdcl_audio, "%s", inData.c_str()));
}

eLogLevel AilAudioSourceImpl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

AudioState AilAudioSourceImpl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    (void)out; //out is not used for capture

    /* calculate number of bytes which we got */
    const int num_received = frames * (mConfig.mNumofChannels * (mConfig.mSampleFormat / NUMOFBITSPERBYTE));
    /* calculate how many bytes can be copied before we exceed the limit defined by VR_PACKET_SIZE */
    int num_copied = (captureOffset + num_received) <= VR_PACKET_SIZE ? num_received : (VR_PACKET_SIZE - captureOffset);
    /* copy chunk into buffer */
    memcpy(&captureBuff[captureOffset], in, num_copied);
    captureOffset += num_copied;

    if (captureOffset == VR_PACKET_SIZE) {
        /* copy data into shared_ptr to guarantee that data is valid and won't be overwritten */
        std::shared_ptr<unsigned char> captureData(new unsigned char[VR_PACKET_SIZE]);
        memcpy(captureData.get(), &captureBuff[0], VR_PACKET_SIZE);

        // todo timestamp
        CCarLifeLib::getInstance()->sendVRRecordData(captureData.get(), VR_PACKET_SIZE, 0);
        captureOffset = 0;

        /* in case there are data left, copy these data in our buffer */
        if (num_received - num_copied <= VR_PACKET_SIZE) {
            if (num_received != num_copied) {
                memcpy(&captureBuff[captureOffset], &in[num_copied], (num_received - num_copied));
                captureOffset = num_received - num_copied;
            }
        } else {
            LOG_ERROR((bdcl_audio, "Data chunk received from AIL is too large"));
        }
    }

    // todo fill last package with 0 if len < 1024
    return AudioState::CONTINUE;
}

void AilAudioSourceImpl::statistics(const adit::utility::audio::StreamStatistics &status)
{
    (void)status;
    // todo [easy, later] implement body using actual StreamStatistics structure rather than printing its address
}

void AilAudioSourceImpl::eostreaming(const AudioError error)
{
    if (error != AudioError::OK)
    {
        LOG_ERROR((bdcl_audio, "eostreaming(): Streaming has stopped unexpectedly: %d", (uint32_t)error));
        // todo [easy, later] why not invoke an onError callback to MC application?
        // todo initiate shuttingDown from here
    }
}

} } /* namespace adit { namespace bdcl */
